const config = require('config');
const axios = require('axios');
const logger = require('../logger/index.js');

const API_KEY = config.get('credenciales.sendinblue.apiKey');
const COUNTRY_CODE_ARG = "+54";
const ERROR_CODE_DUPLICATE_PARAMETER = 'duplicate_parameter';

let sendinblueClient = axios.create({
    baseURL: config.get('sendingbluBaseUrl'),
    headers: {'api-key': `${API_KEY}`}
});

async function saveContactToList(email, nombres, apellidos, sms, listId){
    sms = parseSms(sms);
    createOrUpdateContact(email, nombres, apellidos, sms)
    .then(function (){
        logger.info(`Usuario ${email} creado/actualizado`);
        addContactToList(email, listId);
    })
    .catch(function (error) {
        logger.error(`Error al crear/actualizar usuario en sendinblue: ${JSON.stringify(error.response.data)}`);
    });
}

function parseSms(sms){
    return sms.includes(COUNTRY_CODE_ARG) ? sms : `${COUNTRY_CODE_ARG}${sms}`;
}

async function createOrUpdateContact(email, nombres, apellidos, sms){
    return await sendinblueClient.post('/contacts', {
        "email": email,
        "attributes": {
            "NOMBRE": nombres,
            "SURNAME": apellidos,
            "SMS": sms
        }
    })
        .then(async function (response) {
            return response.data;
        })
        .catch(async function (error) {
            if(error.response.data.code === ERROR_CODE_DUPLICATE_PARAMETER){
                return await sendinblueClient.put(encodeURI(`/contacts/${email}`), {
                    "attributes": {
                        "NOMBRE": nombres,
                        "SURNAME": apellidos,
                        "SMS": sms
                    }
                })
                .then(async function (response) {
                    return response.data;
                })
            }
            else{
                logger.error(error.data);
                return error;
            }
        });
}

async function addContactToList(email, listId){
    return await sendinblueClient.post(`/contacts/lists/${listId}/contacts/add`, {
        "emails": [email]
    })
        .then(async function (response) {
            logger.info(`Usuario ${email} agregado a lista ${listId}`)
            return response.data;
        })
        .catch(async function (error) {
            logger.error(`Error al agregar al usuario ${email} a la lista ${listId}:`);
            logger.error(error.response.data);
        });
}

module.exports =
    {
        saveContactToList: saveContactToList
    }