const config = require('config');
const axios = require('axios');
let _ = require('lodash');
const logger = require('../logger/index.js');

const AUTH_TOKEN = config.get('credenciales.isApiToken');

let isClient = axios.create({
    baseURL: config.get('isBaseUrl'),
    headers: {'Authorization': `Bearer ${AUTH_TOKEN}`}
});

const CABA = "CIUDAD AUTONOMA DE BUENOS AIRES";
const BUENOS_AIRES = "BUENOS AIRES";
const MISIONES = "MISIONES";

function sortProvinciasByName(provincias){
    //Orden alfabetico, pero con (1) CABA, (2) Buenos Aires y (3) Misiones al principio
    let provinciasByName = _.orderBy(provincias, [provincia => provincia.descripcion], ['asc']);
    provinciasByName = _.sortBy(provinciasByName, ({descripcion}) => descripcion === MISIONES ? 0 : 1);
    provinciasByName = _.sortBy(provinciasByName, ({descripcion}) => descripcion === BUENOS_AIRES ? 0 : 1);
    provinciasByName = _.sortBy(provinciasByName, ({descripcion}) => descripcion === CABA ? 0 : 1);
    return provinciasByName;
}

module.exports =
    {
        getProvincias: async function () {
            return await isClient.get('provincias/atencion')
                .then(async function (response) {
                    let provincias = response.data;
                    return sortProvinciasByName(provincias);
                })
                .catch(function (error) {
                    logger.error(error.data);
                });
        },
        getMedicosByName: async function (nombreMedico, idProvincia) {
            return await isClient.get(`/medicos/atencion?id_provincia=${idProvincia}&nombre=${nombreMedico}&todos=1`)
                .then(async function (response) {
                    return response.data.data;
                })
                .catch(function (error) {
                    logger.error(error.data);
                });
        },
        getEspecialidadesMedico: async function (idPersonaInstitucional, idProvincia) {
            return await isClient.get(`/atenciones?id_provincia=${idProvincia}&id_persona_institucional=${idPersonaInstitucional}&todos=1`)
                .then(async function (response) {
                    return response.data.data;
                })
                .catch(function (error) {
                    logger.error(error.data);
                });
        },
        getTurnosDisponibles: async function (idAgenda, idPersonaInstitucional, fechaUsa) {
            return await isClient.get(`/turnos/turnos_disponibles?id_agenda=${idAgenda}&id_persona_institucional=${idPersonaInstitucional}&fecha=${fechaUsa}`)
                .then(async function (response) {
                    return response.data.turnos;
                })
                .catch(function (error) {
                    logger.error(error.data);
                });
        },
        getSiguientesTurnosDisponibles: async function (idAgenda, idPersonaInstitucional) {
            return await isClient.get(`/turnos/turnos_siguientes_disponibles?id_agenda=${idAgenda}&id_persona_institucional=${idPersonaInstitucional}`)
                .then(async function (response) {
                    return response.data;
                })
                .catch(function (error) {
                    logger.error(error.data);
                });
        },
        existeUsuario: async function(tipo_documento, numero_documento, fecha_nacimiento, sexo){
            return await isClient.get(`/watson/existe_persona?tipo_documento=${tipo_documento}&numero_documento=${numero_documento}&fecha_nacimiento=${fecha_nacimiento}&sexo=${sexo}`)
                .then(async function (response) {
                    return response.data;
                })
                .catch(function (error) {
                    logger.error(error.data);
                });
        },
        crearTurno: async function (idPersonaFederada, idHorario, fechaHora) {
            return await isClient.post('/watson/confirmar_reserva', {
                "id_persona_federada": idPersonaFederada,
                "id_horario": idHorario,
                "fecha_hora": fechaHora
            })
                .then(async function (response) {
                    return response.data;
                })
                .catch(function (error) {
                    logger.error(error.data);
                    return error;
                });
        },
        federarUsuario: async function (tipoDocumento, numeroDocumento, fechaNacimiento, sexo, mail, telefonoCelular, idDominio) {
            let data = {
                tipo_documento: tipoDocumento,
                numero_documento: numeroDocumento,
                fecha_nacimiento: fechaNacimiento,
                sexo: sexo,
                mail: mail,
                telefono_celular: telefonoCelular
            };

            if(idDominio !== null){
                data.id_dominio = idDominio;
            }

            return await isClient.post(`/watson/federar_persona`, data)
                .then(async function (response) {
                    return response.data;
                })
                .catch(function (error) {
                    logger.error(error.data);
                    return {success: false};
                });
        },
        crearTurnoBajoDemanda: async function (idAgenda, idPersonaFederada) {
            return await isClient.post(`/watson/confirmar_reserva_espontanea?id_agenda=${idAgenda}&id_persona_federada=${idPersonaFederada}`)
                .then(async function (response) {
                    return response.data;
                })
                .catch(function (error) {
                    logger.error(error.data);
                    return {success: false};
                });
        }
    }