const crisp = require('./crisp.js');
const utils = require('./utils.js');
const pasos = require('./pasos.js');
const is = require('./endpoints/is.js')

let LINK_SOLICITAR_ACCESO_POR_EMAIL = 'https://www.integrandosalud.com/es-ar/integrando-pacientes-solicitar-acceso-por-email/';

let OPTIONS_LIMIT = 4;
let VER_MAS_OPTION_ID = '99';
let MOSTRANDO_PROVINCIAS_HASTA_KEY = 'mostrando_provincias_hasta'
let MOSTRANDO_MEDICOS_ESPECIALIDAD_HASTA_KEY = 'mostrando_medicos_hasta'
let MOSTRANDO_TURNOS_HASTA_KEY = 'mostrando_turnos_hasta'
let ID_PROVINCIA = 'id_provincia'
let ID_AGENDA = 'id_agenda'
let ID_PERSONA_INSTITUCIONAL = 'id_persona_institucional'
let FECHA_TURNO = 'fecha_turno'
let NOMBRE_MEDICO_INPUT = 'nombre_medico_input'
let NOMBRE_MEDICO = 'nombre_medico'
let SIGUIENTE_PASO = 'siguiente_paso'

let MEDICOS = 'medicos'
let MEDICOS_ESPECIALIDAD = 'medicos_especialidad'
let SIGUIENTES_TURNOS_DISPONIBLES = 'siguientes_turnos_disponibles'
let TURNOS_DISPONIBLES = 'turnos_disponibles'
let TURNO_ELEGIDO = 'turno_elegido'
let TIPO_DOCUMENTO = 'tipo_documento'
let NUMERO_DOCUMENTO = 'numero_documento'
let FECHA_NACIMIENTO = 'fecha_nacimiento'
let SEXO = 'sexo'
let CORREO = 'correo'

function getMostrandoDesde(userData, hastaKey) {
    let desde = 0;
    if (userData[hastaKey] != null) {
        desde = userData[hastaKey];
    }
    return desde;
}

async function getSiguientesProvinciasMostrar(userData) {
    let provincias = await is.getProvincias();
    let choices = [];
    let desde = getMostrandoDesde(userData, MOSTRANDO_PROVINCIAS_HASTA_KEY);
    let iter = desde;
    for (iter; iter < desde + OPTIONS_LIMIT && iter < provincias.length; iter++) {
        choices.push(utils.armarChoice(provincias[iter].id_provincia, provincias[iter].descripcion));
    }
    if(iter !== provincias.length){
        choices.push(utils.armarChoice(VER_MAS_OPTION_ID, 'Ver más provincias'));
    }
    return choices;
}

async function getSiguientesMedicosMostrar(userData){
    let medicos_especialidad = [];
    if(userData[MEDICOS_ESPECIALIDAD] != null){
        medicos_especialidad = userData[MEDICOS_ESPECIALIDAD];
    } else{
        let medicos = userData[MEDICOS];
        for(let i=0; i<medicos.length; i++){
            //Por cada medico, obtenemos su especialidad
            let id_persona_institucional = medicos[i].id_persona_institucional;
            let medico_especialidades = await is.getEspecialidadesMedico(id_persona_institucional, userData[ID_PROVINCIA]);
            //Y agrupamos los datos del medico
            for(let j=0; medico_especialidades != null && j<medico_especialidades.length; j++){
                let medico = Object.create(medicos[i]);
                medico.especialidad = medico_especialidades[j].especialidad;
                medico.agendaId = medico_especialidades[j].agendaId;
                medico.institucion = medico_especialidades[j].institucion;
                medico.videoconsultas = medico_especialidades[j].videoconsultas ? 'Si' : 'No';
                medicos_especialidad.push(medico);
            }
        }
        userData[MEDICOS_ESPECIALIDAD] = medicos_especialidad;
    }

    let desde = getMostrandoDesde(userData, MOSTRANDO_MEDICOS_ESPECIALIDAD_HASTA_KEY);

    //Armamos picker
    let choices = [];
    let iter = desde;
    for(iter; iter < desde + OPTIONS_LIMIT && iter < medicos_especialidad.length; iter++){
        let medico_especialidad = `${medicos_especialidad[iter].nombre} - Especialidad: ${medicos_especialidad[iter].especialidad} - Institucion: ${medicos_especialidad[iter].institucion} - Atiende Videoconsulta: ${medicos_especialidad[iter].videoconsultas}`;
        choices.push(utils.armarChoice(medicos_especialidad[iter].agendaId.toString(), medico_especialidad));
    }

    if(iter !== medicos_especialidad.length){
        choices.push(utils.armarChoice(VER_MAS_OPTION_ID, 'Ver más'));
    }

    return choices;
}

async function getSiguientesTurnosMostrar(website_id, session_id, userData){
    let turnos;
    if(userData[TURNOS_DISPONIBLES] != null){
        turnos = userData[TURNOS_DISPONIBLES];
    } else{
        turnos = await is.getTurnosDisponibles(userData[ID_AGENDA], userData[ID_PERSONA_INSTITUCIONAL], fechaArgToFechaUsa(userData[FECHA_TURNO]));
    }

    let desde = getMostrandoDesde(userData, MOSTRANDO_TURNOS_HASTA_KEY);
    let iter = desde;
    let choices = [];
    for(iter; iter < desde + OPTIONS_LIMIT && iter < turnos.length; iter++){
        let fechaHoraUsa = turnos[iter].fechaHora;
        let fechaHoraArg = fechaHoraUsaToFechaHoraArg(fechaHoraUsa);
        let horaArg = fechaHoraArg.split(" ")[1];
        choices.push(utils.armarChoice(iter.toString(), horaArg));
    }

    if(iter !== turnos.length){
        choices.push(utils.armarChoice(VER_MAS_OPTION_ID, 'Ver más'));
    }

    userData[TURNOS_DISPONIBLES] = turnos;
    await crisp.updateUserData(website_id, session_id, userData);
    return choices;
}

function updateChoicesMostradas(userData, hastaKey) {
    let desde = getMostrandoDesde(userData, hastaKey);
    userData[hastaKey] = desde + OPTIONS_LIMIT;
    return userData;
}

function getChoiceElegida(message){
    for (let i = 0; i < message.content.choices.length; i++) {
        if (message.content.choices[i].selected === true) {
            return message.content.choices[i];
        }
    }
}

function fechaArgToFechaUsa(fechaArg){
    let year = fechaArg.split('/')[2];
    let month = fechaArg.split('/')[1];
    let day = fechaArg.split('/')[0];
    let fechaUSA = year + "-" + month + "-" + day;
    return fechaUSA;
}

function fechaHoraUsaToFechaHoraArg(fechaHoraUsa){
    let fechaUsa = fechaHoraUsa.split(" ")[0];
    let horaUsa = fechaHoraUsa.split(" ")[1];

    let year = fechaUsa.split("-")[0];
    let month = fechaUsa.split("-")[1];
    let day = fechaUsa.split("-")[2];

    let hora = horaUsa.split(":")[0];
    let minuto = horaUsa.split(":")[1];
    let fechaHoraArg = `${day}/${month}/${year} ${hora}:${minuto}`;
    return fechaHoraArg;
}

function fechaHoraArgToFechaHoraUsa(fechaHoraArg){
    let fechaArg = fechaHoraArg.split(" ")[0];
    let fechaUsa = fechaArgToFechaUsa(fechaArg);

    let horaArg = fechaHoraArg.split(" ")[1];
    let hora = horaArg.split(":")[0];
    let minuto = horaArg.split(":")[1];

    let fechaHoraUsa = `${fechaUsa} ${hora}:${minuto}:00`;
    return fechaHoraUsa;
}

function getSiguientesFechasConTurnos(website_id, session_id, userData){
    let siguientesTurnos = userData[SIGUIENTES_TURNOS_DISPONIBLES];
    let turnos = siguientesTurnos.turnos;
    let choices = [];
    let fechasAgregadas = {};
    for(let i = 0; i < OPTIONS_LIMIT && i < turnos.length; i++){
        let fechaHoraUsa = turnos[i].fechaHora;
        let fechaHoraArg = fechaHoraUsaToFechaHoraArg(fechaHoraUsa);
        let fechaArg = fechaHoraArg.split(" ")[0];
        //Si la misma fecha ya no fue agregada, agregarla
        if(fechasAgregadas[fechaArg] == null){
            fechasAgregadas[fechaArg] = fechaArg;
            choices.push(utils.armarChoice(i.toString(), fechaArg));
        }
    }
    return choices;
}

module.exports =
    {
        ask_reservar_turno: async function (message, userData, website_id, session_id) {
            choices = [];
            choices.push(utils.armarChoice("1", "Si"));
            choices.push(utils.armarChoice("2", "No"));
            await crisp.setSiguientePaso(website_id, session_id, pasos.ASK_PROVINCIA, userData);
            await crisp.sendPickerMessage(website_id, session_id, pasos.ASK_PROVINCIA, "Hola! Puedo ayudarte a reservar un turno, ¿te gustaría hacerlo?", choices);
        },
        ask_provincia: async function (message, userData, website_id, session_id) {
            if (message.content.choices[0].selected === true) {
                let choices = await getSiguientesProvinciasMostrar(userData);
                userData = updateChoicesMostradas(userData, MOSTRANDO_PROVINCIAS_HASTA_KEY);
                await crisp.setSiguientePaso(website_id, session_id, pasos.RESP_PROVINCIA, userData);
                await crisp.sendPickerMessage(website_id, session_id, pasos.RESP_PROVINCIA, "¿En qué provincia se encuentra tu médico?", choices);
            } else {
                await crisp.sendTextMessage(website_id, session_id, "Gracias por contactarte! Estoy acá para cuando quieras reservar un turno");
            }
        },
        resp_provincia: async function (message, userData, website_id, session_id) {
            let choiceElegida = getChoiceElegida(message);
            if (choiceElegida.value === VER_MAS_OPTION_ID) {
                //Ver mas provincias
                let choices = await getSiguientesProvinciasMostrar(userData);
                userData = updateChoicesMostradas(userData, MOSTRANDO_PROVINCIAS_HASTA_KEY);
                await crisp.setSiguientePaso(website_id, session_id, pasos.RESP_PROVINCIA, userData);
                await crisp.sendPickerMessage(website_id, session_id, pasos.RESP_PROVINCIA, "¿En qué provincia se encuentra tu médico?", choices);
            } else {
                //Set provincia y avanzar
                userData[ID_PROVINCIA] = choiceElegida.value;
                await this.ask_nombre_medico(message, userData, website_id, session_id);
            }
        },
        ask_nombre_medico: async function(message, userData, website_id, session_id){
            userData[SIGUIENTE_PASO] = pasos.RESP_NOMBRE_MEDICO;
            await crisp.updateUserData(website_id, session_id, userData);
            await crisp.sendTextMessage(website_id, session_id,"¿Cuál es el apellido de tu médico?");
        },
        resp_nombre_medico: async function(message, userData, website_id, session_id){
            let nombreMedico = message.content;
            userData[NOMBRE_MEDICO_INPUT] = nombreMedico;
            let medicos = await is.getMedicosByName(userData[NOMBRE_MEDICO_INPUT], userData[ID_PROVINCIA]);
            if(medicos.length > 0){
                userData[MEDICOS] = medicos;
                await crisp.updateUserData(website_id, session_id, userData);
                await this.ask_medico_especialidad(message, userData, website_id, session_id);
            }
            else{
                await crisp.sendTextMessage(website_id, session_id,`No encuentro médicos con el nombre ${nombreMedico} en la provincia`);
                await crisp.sendTextMessage(website_id, session_id, "Por favor, reintente ingresar el apellido");
                await crisp.sendTextMessage(website_id, session_id,"¿Cuál es el apellido de tu médico?");
            }
        },
        ask_medico_especialidad: async function(message, userData, website_id, session_id){
            let choices = await getSiguientesMedicosMostrar(userData);
            userData = updateChoicesMostradas(userData, MOSTRANDO_MEDICOS_ESPECIALIDAD_HASTA_KEY);
            await crisp.sendPickerMessage(website_id, session_id, pasos.RESP_MEDICO_ESPECIALIDAD, "Seleccione uno de los siguientes medicos, según su especialidad", choices);
            await crisp.setSiguientePaso(website_id, session_id, pasos.RESP_MEDICO_ESPECIALIDAD, userData);
        },
        resp_medico_especialidad: async function(message, userData, website_id, session_id){
            let choiceElegida = getChoiceElegida(message);
            if(choiceElegida.value === VER_MAS_OPTION_ID){
                //Mostrar mas medicos
                let choices = await getSiguientesMedicosMostrar(userData);
                userData = updateChoicesMostradas(userData, MOSTRANDO_MEDICOS_ESPECIALIDAD_HASTA_KEY);
                await crisp.updateUserData(website_id, session_id, userData)
                await crisp.sendPickerMessage(website_id, session_id, pasos.RESP_MEDICO_ESPECIALIDAD, "Seleccione uno de los médicos encontrados, según su especialidad", choices);
            }
            else{
                //Set medico-especialidad y avanzar
                userData[ID_AGENDA] = parseInt(choiceElegida.value);
                userData[NOMBRE_MEDICO] = choiceElegida.label.split(' -')[0]; //Quitamos la especialidad
                //Set id_persona_institucional
                let medicos_especialidad = userData[MEDICOS_ESPECIALIDAD];
                for(let i=0; i<medicos_especialidad.length; i++){
                    if(medicos_especialidad[i].agendaId === userData[ID_AGENDA]){
                        userData[ID_PERSONA_INSTITUCIONAL] = medicos_especialidad[i].id_persona_institucional;
                    }
                }
                await this.ask_fecha_turno(message, userData, website_id, session_id);
            }
        },
        ask_fecha_turno: async function(message, userData, website_id, session_id){
            //Verificar que el medico tenga turnos disponibles
            let siguientesTurnos = await is.getSiguientesTurnosDisponibles(userData[ID_AGENDA], userData[ID_PERSONA_INSTITUCIONAL]);
            userData[SIGUIENTES_TURNOS_DISPONIBLES] = siguientesTurnos;
            if(siguientesTurnos.turnos.length === 0){
                let telefonoMedico = siguientesTurnos.tel;
                await crisp.sendTextMessage(website_id, session_id, `Disculpe, el doctor ${userData[NOMBRE_MEDICO]} no tiene turnos proximos disponibles`);
                await crisp.sendTextMessage(website_id, session_id,`Le sugiero llamar al teléfono ${telefonoMedico} para coordinar un turno telefónicamente`);
                await crisp.sendTextMessage(website_id, session_id,`Si desea reservar un turno con otro médico, por favor escriba: REINICIAR`);
            } else {
                await crisp.setSiguientePaso(website_id, session_id, pasos.RESP_FECHA_TURNO, userData);
                await crisp.sendTextMessage(website_id, session_id, `Cuándo querrías atenderte con ${userData[NOMBRE_MEDICO]}? (DD/MM/AAAA)`);
            }
        },
        resp_fecha_turno: async function(message, userData, website_id, session_id){
            userData[FECHA_TURNO] = message.content;
            let siguientesTurnos = userData[SIGUIENTES_TURNOS_DISPONIBLES];
            let turnos = siguientesTurnos.turnos;
            let existeTurnoEnFecha = false;
            for(let i=0; i < turnos.length; i++){
                let fechaHoraUsa = turnos[i].fechaHora;
                let fechaUsa = fechaArgToFechaUsa(userData[FECHA_TURNO]);
                if(fechaHoraUsa.includes(fechaUsa)){
                    existeTurnoEnFecha = true;
                }
            }
            if(existeTurnoEnFecha){
                await this.ask_hora_turno(message, userData, website_id, session_id);
            }
            else{
                //Mostrar las siguientes fechas con turnos
                let choices = await getSiguientesFechasConTurnos(website_id, session_id, userData);
                let fechaArg = userData[FECHA_TURNO];
                await crisp.sendPickerMessage(website_id, session_id, pasos.RESP_FECHA_TURNO_SIGUIENTES_FECHAS,
                    `Disculpe, no hay turnos para el ${fechaArg}, pero hay turnos disponibles en las siguientes fechas`, choices);
                await crisp.setSiguientePaso(website_id, session_id, pasos.RESP_FECHA_TURNO_SIGUIENTES_FECHAS, userData);
            }
        },
        resp_fecha_turno_siguientes_fechas: async function(message, userData, website_id, session_id){
            let choiceElegida = getChoiceElegida(message);
            userData[FECHA_TURNO] = choiceElegida.label;
            await this.ask_hora_turno(message, userData, website_id, session_id);
        },
        ask_hora_turno: async function(message, userData, website_id, session_id){
            let choices = await getSiguientesTurnosMostrar(website_id, session_id, userData);
            userData = updateChoicesMostradas(userData, MOSTRANDO_TURNOS_HASTA_KEY);
            let fechaArg = userData[FECHA_TURNO];
            await crisp.sendPickerMessage(website_id, session_id, pasos.RESP_HORA_TURNO, `Seleccione un turno para el ${fechaArg}`, choices);
            await crisp.setSiguientePaso(website_id, session_id, pasos.RESP_HORA_TURNO, userData);
        },
        resp_hora_turno: async function(message, userData, website_id, session_id){
            let choiceElegida = getChoiceElegida(message);
            if(choiceElegida.value === VER_MAS_OPTION_ID){
                //Mostrar mas turnos
                let choices = await getSiguientesTurnosMostrar(website_id, session_id, userData);
                userData = updateChoicesMostradas(userData, MOSTRANDO_TURNOS_HASTA_KEY);
                let fechaArg = userData[FECHA_TURNO];
                await crisp.updateUserData(website_id, session_id, userData)
                await crisp.sendPickerMessage(website_id, session_id, pasos.RESP_HORA_TURNO, `Seleccione un turno para el ${fechaArg}`, choices);
            }
            else{
                //Set turno elegido y avanzar
                let turnosDisponibles = userData[TURNOS_DISPONIBLES];
                let turno;
                for(let i=0; i < turnosDisponibles.length; i++){
                    let horaArg = choiceElegida.label;
                    let fechaHoraArg = `${userData[FECHA_TURNO]} ${horaArg}`;
                    let fechaHoraUsa = fechaHoraArgToFechaHoraUsa(fechaHoraArg);
                    if(fechaHoraUsa === turnosDisponibles[i].fechaHora){
                        turno = turnosDisponibles[i];
                    }
                }
                userData[TURNO_ELEGIDO] = turno;
                await this.ask_confirmar_fecha_hora(message, userData, website_id, session_id);
            }
        },
        ask_confirmar_fecha_hora: async function(message, userData, website_id, session_id){
            await crisp.sendTextMessage(website_id, session_id, 'Perfecto, confirmemos los datos');
            let fechaArg = userData[FECHA_TURNO];
            let horaArg = fechaHoraUsaToFechaHoraArg(userData[TURNO_ELEGIDO].fechaHora).split(" ")[1];
            await crisp.sendTextMessage(website_id, session_id, `El turno sera programado con ${userData[NOMBRE_MEDICO]}, el ${fechaArg} a las ${horaArg}`);
            let choices = [];
            choices.push(utils.armarChoice('Confirmar', 'Confirmar'));
            choices.push(utils.armarChoice('Modificar', 'Modificar'));
            await crisp.setSiguientePaso(website_id, session_id, pasos.RESP_CONFIRMAR_FECHA_HORA, userData);
            await crisp.sendPickerMessage(website_id, session_id, pasos.RESP_CONFIRMAR_FECHA_HORA, 'Es correcto?', choices);
        },
        resp_confirmar_fecha_hora: async function(message, userData, website_id, session_id){
            let choice = getChoiceElegida(message);
            if(choice.value === 'Confirmar'){
                if(userData[TIPO_DOCUMENTO] == null){
                    await this.ask_tipo_documento(message, userData, website_id, session_id);
                } else{
                    let message = {}
                    message.content = userData[CORREO];
                    await this.resp_correo(message, userData, website_id, session_id);
                }
            }
            else{
                userData[MOSTRANDO_TURNOS_HASTA_KEY] = null;
                userData[TURNOS_DISPONIBLES] = null;
                await this.ask_fecha_turno(message, userData, website_id, session_id);
            }
        },
        ask_tipo_documento: async function(message, userData, website_id, session_id){
            await crisp.sendTextMessage(website_id, session_id, 'Perfecto, ahora vamos a asociar tu datos. Selecciona tu tipo de documento');
            let choices = [];
            choices.push(utils.armarChoice("DNI", "DNI"));
            choices.push(utils.armarChoice("CDM", "CDM"));
            choices.push(utils.armarChoice("CED", "CED"));
            choices.push(utils.armarChoice("CIE", "CIE"));
            choices.push(utils.armarChoice("CM", "CM"));
            await crisp.sendPickerMessage(website_id, session_id, pasos.RESP_TIPO_DOCUMENTO, "Tipo de documento", choices);
            await crisp.setSiguientePaso(website_id, session_id, pasos.RESP_TIPO_DOCUMENTO, userData);
        },
        resp_tipo_documento: async function(message, userData, website_id, session_id){
            let tipoDocumentoElegido = getChoiceElegida(message);
            userData[TIPO_DOCUMENTO] = tipoDocumentoElegido.value;
            await this.ask_numero_documento(message, userData, website_id, session_id);
        },
        ask_numero_documento: async function(message, userData, website_id, session_id){
            await crisp.setSiguientePaso(website_id, session_id, pasos.RESP_NUMERO_DOCUMENTO, userData);
            await crisp.sendTextMessage(website_id, session_id, 'Ingresa tu número de documento');
        },
        resp_numero_documento: async function(message, userData, website_id, session_id){
            userData[NUMERO_DOCUMENTO] = message.content;
            await this.ask_fecha_nacimiento(message, userData, website_id, session_id);
        },
        ask_fecha_nacimiento: async function(message, userData, website_id, session_id){
            await crisp.setSiguientePaso(website_id, session_id, pasos.RESP_FECHA_NACIMIENTO, userData);
            await crisp.sendTextMessage(website_id, session_id, 'Ingresa tu fecha de nacimiento (DD/MM/AAAA)');
        },
        resp_fecha_nacimiento: async function(message, userData, website_id, session_id){
            userData[FECHA_NACIMIENTO] = message.content;
            await this.ask_sexo(message, userData, website_id, session_id);
        },
        ask_sexo: async function(message, userData, website_id, session_id){
            let choices = [];
            choices.push(utils.armarChoice("M", "Masculino"));
            choices.push(utils.armarChoice("F", "Femenino"));
            await crisp.setSiguientePaso(website_id, session_id, pasos.RESP_SEXO, userData);
            await crisp.sendPickerMessage(website_id, session_id, pasos.RESP_SEXO, 'Selecciona tu sexo', choices);
        },
        resp_sexo: async function(message, userData, website_id, session_id){
            userData[SEXO] = getChoiceElegida(message).value;
            await this.ask_correo(message, userData, website_id, session_id);
        },
        ask_correo: async function(message, userData, website_id, session_id){
            await crisp.setSiguientePaso(website_id, session_id, pasos.RESP_CORREO, userData);
            await crisp.sendTextMessage(website_id, session_id, 'Ingresa tu correo electrónico');
        },
        resp_correo: async function(message, userData, website_id, session_id){
            userData[CORREO] = message.content;
            await crisp.updateCrispData(website_id, session_id, userData[CORREO], userData[TIPO_DOCUMENTO], userData[NUMERO_DOCUMENTO], userData[FECHA_NACIMIENTO], userData[SEXO]);
            if(!await is.existeUsuario(userData[TIPO_DOCUMENTO], userData[NUMERO_DOCUMENTO], userData[FECHA_NACIMIENTO], userData[SEXO]) === true){
                await crisp.sendTextMessage(website_id, session_id,
                    'No se pudieron validar sus datos. Solicite por favor su acceso por email ingresando al siguiente link'
                );
                await crisp.sendTextMessage(website_id, session_id, LINK_SOLICITAR_ACCESO_POR_EMAIL);
                return;
            }
            let turnoElegido = userData[TURNO_ELEGIDO];
            let resp = await is.crearTurno(userData[TIPO_DOCUMENTO], userData[NUMERO_DOCUMENTO], fechaArgToFechaUsa(userData[FECHA_NACIMIENTO]),
                userData[SEXO], userData[CORREO], 0, turnoElegido.horarioId, turnoElegido.fechaHora);
            if(resp.success === true){
                let fechaHoraArg = fechaHoraUsaToFechaHoraArg(turnoElegido.fechaHora);
                let fechaArg = fechaHoraArg.split(" ")[0];
                let horaArg = fechaHoraArg.split(" ")[1];
                await crisp.sendTextMessage(website_id, session_id,
                    `Listo! El turno fue reservado para el ${fechaArg} a las ${horaArg}, con ${userData[NOMBRE_MEDICO]}`
                );
                if(resp.link_mercadopago != null){
                    await crisp.sendTextMessage(website_id, session_id,'Haz click aquí si deseas abonar la consulta en este momento');
                    await crisp.sendTextMessage(website_id, session_id, resp.link_mercadopago);
                }
                await this.ask_reservar_otro_turno(message, userData, website_id, session_id);
            }
            else{
                await crisp.sendTextMessage(website_id, session_id,'Ha ocurrido un error al reservar el turno, intente nuevamente');
                await crisp.updateUserData(website_id, session_id, {});
                await this.ask_reservar_turno(message, userData, website_id, session_id);
            }
        },
        ask_reservar_otro_turno: async function(message, userData, website_id, session_id){
            let choices = [];
            choices.push(utils.armarChoice("Si", "Si"));
            choices.push(utils.armarChoice("No", "No"));
            await crisp.sendPickerMessage(website_id, session_id, pasos.RESP_RESERVAR_OTRO_TURNO, "¿Quieres reservar otro turno?", choices);
            await crisp.setSiguientePaso(website_id, session_id, pasos.RESP_RESERVAR_OTRO_TURNO, userData);
        },
        resp_reservar_otro_turno: async function(message, userData, website_id, session_id){
            let choice = getChoiceElegida(message);
            if(choice.label === 'No'){
                await crisp.sendTextMessage(website_id, session_id,'Estoy acá para cuando necesites reservar otro turno!');
                await crisp.updateUserData(website_id, session_id, {});
            }
            else{
                userData[MOSTRANDO_MEDICOS_ESPECIALIDAD_HASTA_KEY] = null;
                userData[MOSTRANDO_TURNOS_HASTA_KEY] = null;
                userData[MEDICOS_ESPECIALIDAD] = null;
                userData[MEDICOS] = null;
                userData[TURNOS_DISPONIBLES] = null;
                await crisp.updateUserData(website_id, session_id, userData);
                await this.ask_nombre_medico(message, userData, website_id, session_id);
            }
        }
    }